unit EngineImgLoadSave01;
(*
   ========================================================================
    " " GraphEngine.
       .
   : BitMap.PixelFormat = pf24bit
   ========================================================================
       TImageDesc  TLoadsaveImage:
   1)    (JPEG  BMP)  Image  .
   2)     Image   Clipboard.
   ========================================================================
   ()  ,    , , .
   ========================================================================
*)

interface
uses
     //  
     Windows, Classes, Controls, Graphics, Dialogs, Forms,
     SysUtils, ExtDlgs, ComCtrls, ExtCtrls, JPEG, Clipbrd,
     //  GraphEngine
     EngineMainData01;

// ------------------------------------------------------------------------
//    
const FileFormatJPEG  = 1;     //    JPEG
      FileFormatBMP   = 2;     //    BMP
      ClipboardFormat = 128;   //    BitMap Clipboard
      FileFormatUnkn  = 255;   //    

// ------------------------------------------------------------------------
// ------------------------------------------------------------------------
//     Image (  Image.Tag)
type TImageDesc = class(Tobject)
  private
   fImageName    : string;        //     Image
   fFileExtCode  : byte;          //   
   fFilePixCode  : TPixelFormat;  //    
 public
   //   
   property ImageFileName : string read fImageName write fImageName;
   //   
   property FileExtCode : byte read fFileExtCode write fFileExtCode;
   // BitMap.PixelFormat  
   property FilePixCode : TPixelFormat read fFilePixCode write fFilePixCode;
end;

// ------------------------------------------------------------------------
type TLoadSaveImage = class(Tobject)
  private
     fInitDir     : string;               //   
     fForm        : TForm;
     //    
     fLoadDlg     : TOpenPictureDialog;   //   
     fSaveDlg     : TSavePictureDialog;   //   
     //     
     fProgressBar : TProgressBar;         //   ProgressBar
     fMsgPanel    : TStatusPanel;         //   StatusPanel
     //    JPEG
     fJpegQuality     : byte;             // CompressionQuality  JPEG
     fJpegProgressive : boolean;          //   JPEG
     //   
     //    
     function VerifyPixFormat (RqBitMap : TBitMap) : boolean;
     //    
     function GetIdPixFormat (RqBitMap : TBitMap) : string;
     //       BitMap
     function LoadJpegPicFromFile (RqFileName  : string;
                                   RqImage     : TImage) : boolean;
     //  BMP   Image.Pictures.BitMap
     function LoadBmpPicFromFile  (RqFileName  : string;
                                   RqImage     : TImage) : boolean;
     //  Image.Pictures.BitMap  JPEG    
     function SaveJpegToFile (RqFileName : string;
                              RqImage    : TImage) : boolean;
     //    Image.Picture.Bitmap  
     function SaveBmpToFile  (RqFileName    : string;
                              RqImage       : TImage) : boolean;
  public
     //    
     constructor Create (RqProgressBar : TProgressBar;  //   ProgressBar
                         RqMsgPanel    : TStatusPanel); //   StatusPanel
     //  
     procedure Free;
     //   Image     Image  
     procedure SetImageDescriptor
                      (RqImage        : TImage;
                       RqFileName     : string;          //  
                       RqFileExtCode  : byte;            //  
                       RqFilePixCode  : TPixelFormat);   //  
     //  Image - File
     //       Image
     function LoadImgFromFile (RqImg : TImage) : boolean;
     //     Image  
     function SaveImgToFile (RqImg : TImage) : boolean;
     //  Image - Clipboard
     //   Image  Clipboard
     function CopyImageToClipboard(RqImg : TImage) : boolean;
     //   Clipboard  Image
     function PasteClipboardToImage(RqImage : TImage) : boolean;
     // 
     //    (  )
     property wForm : TForm read fForm  write fForm;
     //    JPEG
     property JpegQuality : byte read fJpegQuality
                                 write fJpegQuality;
     property JpegProgressive : boolean read  fJpegProgressive
                                        write fJpegProgressive;

end;

// ========================================================================
// ========================================================================
implementation
// ========================================================================
// ========================================================================

//       FileFormatJPEG  FileFormatBMP
// ( !)
const DialogFilter = 'JPEG (*.jpg;*.jpeg)|*.jpg;*.jpeg | BMP (*.bmp)|*.bmp';

// ========================================================================
//     
// ========================================================================
//  
constructor TLoadsaveImage.Create
                  (RqProgressBar : TProgressBar;   //   ProgressBar
                   RqMsgPanel    : TStatusPanel);  //   StatusPanel;
begin
  inherited Create;
  //     
  fProgressBar := RqProgressBar;  //   ProgressBar
  fMsgPanel    := RqMsgPanel;     //   StatusPanel
  fForm := nil;                   //    
  //    OpenPictureDialog
  fLoadDlg := TOpenPictureDialog.Create(nil);
  with fLoadDlg do
  begin
    Filter := DialogFilter;
    FilterIndex := FileFormatJPEG;
    Ctl3D := True;
  end;
  fInitDir := '';                //   
  //    SavePictureDialog
  fSaveDlg := TSavePictureDialog.Create(nil);
  with fSaveDlg do
  begin
    Filter := DialogFilter;
    FilterIndex := FileFormatJPEG;
    Ctl3D := True;
  end;
  //     JPEG
  fJpegQuality := 100;
  fJpegProgressive := False;
end;
// ------------------------------------------------------------------------
//  
procedure TLoadsaveImage.Free;
begin
   fLoadDlg.FreeOnRelease;
   fSaveDlg.FreeOnRelease;
   inherited Free;
end;

// ========================================================================
//  PUBLIC   
// ========================================================================
//      Image
function TLoadsaveImage.LoadImgFromFile (RqImg : TImage) : boolean;
begin
  Result := False;
  if fInitDir <> '' then fLoadDlg.InitialDir := fInitDir;
  if fLoadDlg.Execute
  then begin
    if Assigned(fForm)
    then begin
      Screen.Cursor := crHourGlass;
      fForm.Repaint;
    end;
    //    
    fInitDir := ExtractFileDir(fLoadDlg.FileName);
    case fLoadDlg.FilterIndex of
     //  JPEG   Image.Pictures.BitMap
     FileFormatJPEG : Result := LoadJpegPicFromFile (fLoadDlg.FileName, RqImg);
     //  BMP   Image.Pictures.BitMap
     FileFormatBMP  : Result := LoadBmpPicFromFile (fLoadDlg.FileName, RqImg);
    end;
    if Assigned(fForm) then Screen.Cursor := crDefault;
  end;
end;

// ------------------------------------------------------------------------
//    Image  
function TLoadsaveImage.SaveImgToFile (RqImg : TImage) : boolean;
var FileName,             //   
    FileExt : string;     //   
    URep    : word;       //   
begin
   Result := False;
   if fInitDir <> '' then fSaveDlg.InitialDir := fInitDir;
   if fSaveDlg.Execute
   then begin
      if Assigned(fForm)
      then begin
         Screen.Cursor := crHourGlass;
         fForm.Repaint;
      end;
      FileName := fSaveDlg.FileName;
      FileExt  := UpperCase(ExtractFileExt(FileName));
      case fSaveDlg.FilterIndex of
        FileFormatJPEG :  begin
            //     ,    
            if not ((FileExt = '.JPG') or (FileExt = '.JPEG'))
            then FileName := FileName + '.jpg';
            if FileExists(FileName)
            then begin
               URep := MessageDlg('   .'+
                                  '   ?',
                                   mtInformation,[mbYes,mbNo],0);
               //  
               if URep = mrYes
               then Result := SaveJpegToFile (FileName, RqImg);
            end
            else Result := SaveJpegToFile (FileName, RqImg);
        end;
        FileFormatBMP  :  begin
           //     ,    
            if not (FileExt = '.BMP')
            then FileName := FileName + '.bmp';
            if FileExists(FileName)
            then begin
               URep := MessageDlg('   .'+
                                  '   ?',
                                   mtInformation,[mbYes,mbNo],0);
               //  
               if URep = mrYes then Result := SaveBmpToFile (FileName, RqImg);
            end
            else Result := SaveBmpToFile (FileName, RqImg);
        end;
      end;
      if Assigned(fForm) then Screen.Cursor := crDefault;
   end;
end;
// ========================================================================
//  PIVATE   
// ========================================================================
//    
function TLoadsaveImage.VerifyPixFormat (RqBitMap : TBitMap) : boolean;
begin
  Result := False;
  case RqBitMap.PixelFormat of
    pfDevice : Result := True;
    pf4bit   : Result := True;
    pf8bit   : Result := True;
    pf15bit  : Result := True;
    pf16bit  : Result := True;
    pf24bit  : Result := True;
    pf32bit  : Result := True;
  end;
end;
// ------------------------------------------------------------------------
//    
function TLoadsaveImage.GetIdPixFormat (RqBitMap : TBitMap) : string;
begin
  case RqBitMap.PixelFormat of
    pfDevice : Result := 'pfDevice';
    pf1bit   : Result := 'pf1bit';
    pf4bit   : Result := 'pf4bit';
    pf8bit   : Result := 'pf8bit';
    pf15bit  : Result := 'pf15bit';
    pf16bit  : Result := 'pf16bit';
    pf24bit  : Result := 'pf24bit';
    pf32bit  : Result := 'pf32bit';
    pfCustom : Result := 'pfCustom';
    else Result := 'pfCustom';
  end;
end;
// ------------------------------------------------------------------------
//   Image     Image  
procedure TLoadsaveImage.SetImageDescriptor
                   (RqImage        : TImage;
                    RqFileName     : string;          //  
                    RqFileExtCode  : byte;            //  
                    RqFilePixCode  : TPixelFormat);   //  
var  WID  : TImageDesc;     //    Image
begin
   WID := nil;
   //   
   if (RqImage.Tag <> 0)
   then begin
       try
         WID := TImageDesc(RqImage.Tag);
       except
         //    ,   
         RqImage.Tag := 0;
       end;
   end
   else begin
      try
        RqImage.Tag := integer(TImageDesc.Create);
        WID := TImageDesc(RqImage.Tag);
      except
        RqImage.Tag := 0;
      end;
  end;
  //  
  if Assigned(WID) then
  begin
    try
      WID.ImageFileName  := RqFileName;
      WID.FileExtCode    := RqFileExtCode; //   
      WID.FilePixCode    := RqFilePixCode; //     (TPixelFormat)
    except
       RqImage.Tag := 0;
    end;
  end;
end;

// ========================================================================
//       JPEG 
// ========================================================================
//        JPEG
type
   TJPEGService = class(TObject)
     fProgress  : TProgressBar;
     //     JPEG 
     procedure Progress(Sender: TObject; Stage: TProgressStage;
               PercentDone: Byte; RedrawNow: Boolean; const R : TRect;
               const Msg: String);
end;
// ------------------------------------------------------------------------
//      JPEG 
procedure TJPEGService.Progress(Sender: TObject;
          Stage: TProgressStage; PercentDone: Byte;
          RedrawNow: Boolean; const R: TRect;
          const Msg: String);
begin
  case Stage of
  psStarting: begin
               fProgress.Position := 0;
               fProgress.Max      := 100;
             end;
  psEnding:   begin
               fProgress.Position := 0;
             end;
  psRunning:  begin
               fProgress.Position := PercentDone;
             end;
 end;
end;
// ------------------------------------------------------------------------
//  JPEG   Image.Pictures.BitMap
function  TLoadsaveImage.LoadJpegPicFromFile
                   (RqFileName    : string;
                    RqImage       : TImage) : boolean;
var WPicture  : TPicture;       //  
    JS        : TJPEGService;   //  
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage)
  then begin
      //   
      WPicture := TPicture.Create;
      JS       := TJPEGService.Create;
      try
         //  WPicture 
         WPicture.LoadFromFile(RqFileName);
         if WPicture.Graphic is TJpegImage then
         begin
           if fProgressBar <> nil
           then begin
               //        JPEG
                JS.fProgress := fProgressBar;
                //    
                TJpegImage(WPicture.Graphic).OnProgress := JS.Progress;
           end;
           //  JPG  BitMap
           TJpegImage(WPicture.Graphic).DIBNeeded;
           //  BitMap  ImageMain
           RqImage.Picture.Bitmap.Assign(TBitmap(WPicture.Graphic));
           //     24   
           RqImage.Picture.Bitmap.PixelFormat := pf24bit;
           //       Image  
           SetImageDescriptor (RqImage, RqFileName, FileFormatJPEG, pf24bit);
           //  
           if fMsgPanel <> nil
           then fMsgPanel.Text := '    ';
           Result := True;
         end;
      finally
         //   
         WPicture.Free;
         JS.Free;
      end;
  end;
end;

// ------------------------------------------------------------------------
//  Image.Pictures.BitMap  JPEG    
function TLoadsaveImage.SaveJpegToFile
                       (RqFileName    : string;
                        RqImage       : TImage
                        ) : boolean;
var JI   : TJpegImage;       //  
    JS   : TJPEGService;     //  
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage.Picture.Bitmap)
  then begin
     //   
     JS := TJPEGService.Create;
     JI := TJpegImage.Create;
     try
        if fProgressBar <> nil
        then begin
           JS.fProgress :=  fProgressBar;
           JI.OnProgress := JS.Progress;  //   
        end;
        JI.PixelFormat := jf24bit;        //    
        if fJpegQuality > 100
        then JI.CompressionQuality := 100
        else JI.CompressionQuality := fJpegQuality;
        JI.ProgressiveEncoding := fJpegProgressive;
        JI.Assign(RqImage.Picture.Bitmap);
        JI.SaveToFile(RqFileName);
        //   Image    
        SetImageDescriptor (RqImage, RqFileName, FileFormatJPEG, pf24bit);
        //  
        if fMsgPanel <> nil
        then fMsgPanel.Text := '    ';
        Result := True;
     finally
        //   
        JI.Free;
        JS.Free;
     end;
  end;
end;

// ========================================================================
//       BMP 
// ========================================================================
//  BMP   Image.Pictures.BitMap
function TLoadsaveImage.LoadBmpPicFromFile
                   (RqFileName    : string;
                    RqImage       : TImage) : boolean;
var WBitMap   : TBitMap;        //  
    PixFormat : TPixelFormat;   //     
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage)
  then begin
      //   
      WBitMap   := TBitMap.Create;;
      try
         //  WBitMap 
         WBitMap.LoadFromFile(RqFileName);
         if VerifyPixFormat(WBitMap)
         then begin
            PixFormat := WBitMap.PixelFormat;
            //     pf24bit
            WBitMap.PixelFormat := pf24bit;
            //     
            if WBitMap.PixelFormat = pf24bit
            then begin
             //   Image    
             SetImageDescriptor (RqImage, RqFileName, FileFormatBMP, PixFormat);
             // ----------------------------------------------------------
             //  BitMap  RqImage
             RqImage.Picture.Bitmap.Assign(WBitMap);
             // ----------------------------------------------------------
             //  
             if fMsgPanel <> nil
             then fMsgPanel.Text := '    ';
             Result := True;
           end
           else
             if fMsgPanel <> nil
             then fMsgPanel.Text := '    pf24bit!';
         end
         else
           if fMsgPanel <> nil
           then fMsgPanel.Text := ' !'
                               +  '   ( '
                               +  GetIdPixFormat (WBitMap)
                               +  ' )   ';
      finally
         //   
         WBitMap.Free;
      end;
  end;
end;
// ------------------------------------------------------------------------
//    Image.Picture.Bitmap  
function TLoadsaveImage.SaveBmpToFile
                        (RqFileName    : string;
                         RqImage       : TImage) : boolean;
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage.Picture.Bitmap)
  then begin
    try
      RqImage.Picture.SaveToFile(RqFileName);
      //  
      if fMsgPanel <> nil
      then fMsgPanel.Text := '    ';
      //       Image  
      SetImageDescriptor (RqImage, RqFileName, FileFormatBMP, pf24bit);
      Result := True;
    except
      if fMsgPanel <> nil
      then fMsgPanel.Text := '    ';
    end;
  end;
end;

// ========================================================================
//      Clipboard
// ========================================================================
//   BitMap  Clipboard
function CopyBitMapToClipboard(RqBitMap : TBitMap) : boolean;
begin
  Result := False;
  if not Assigned (RqBitMap) then Exit;
  if (RqBitMap.Width > 0) and (RqBitMap.Height > 0)
  then begin
    try
       Clipboard.Assign(RqBitMap);
       Result := True;
    except
       MessageDlg('CopyBitMapToClipboard : '
                 + '   Clipboard!',
                  mtWarning, [mbOk], 0);
    end;
  end;
end;
// ------------------------------------------------------------------------

(*
//   Image  Clipboard
function CopyImageToClipboard(RqImg : TImage) : boolean;
begin
  Result := False;
  if Assigned(RqImg.Picture.Bitmap)
  then begin
    Result := CopyBitMapToClipboard (RqImg.Picture.Bitmap);
  end;
end;
*)

// ------------------------------------------------------------------------
//   Image  Clipboard
function TLoadsaveImage.CopyImageToClipboard(RqImg : TImage) : boolean;
begin
  Result := False;
  if Assigned(RqImg.Picture.Bitmap)
  then begin
    Result := CopyBitMapToClipboard (RqImg.Picture.Bitmap);
  end;
end;
// ------------------------------------------------------------------------
//   Clipboard  Image
function TLoadsaveImage.PasteClipboardToImage(RqImage : TImage) : boolean;
var WBitmap    : TBitmap;          //  Bitmap
    PixFormat  : TPixelFormat;     //    Clipboard
begin
  Result := False;
  //   Clipboard  Windows BitMap
  if Clipboard.HasFormat(CF_BITMAP)
  then begin
    WBitmap := TBitmap.Create;
    try
      //  Clipboard  BitMap.
      WBitmap.Assign(Clipboard);
      if VerifyPixFormat(WBitMap)
      then begin
         PixFormat := WBitMap.PixelFormat;
         //     pf24bit
         WBitmap.PixelFormat := pf24bit;
         //     
         if WBitmap.PixelFormat = pf24bit
         then begin
            //  BitMap  RqImage
            RqImage.Picture.Bitmap.Assign(WBitmap);
            //   Image    
            SetImageDescriptor (RqImage, 'Clipboard',
                                 ClipboardFormat, PixFormat);
            if fMsgPanel <> nil
            then fMsgPanel.Text := '     Clipboard';
            Result := True;
         end
         else
            if fMsgPanel <> nil
            then fMsgPanel.Text := '    pf24bit!';
      end
      else
         if fMsgPanel <> nil
         then fMsgPanel.Text := ' !'
                             +  '   ( '
                             +  GetIdPixFormat (WBitMap)
                             +  ' )   Clipboard';
    finally
      WBitmap.Free;
    end;
  end
  else begin
    if fMsgPanel <> nil
    then fMsgPanel.Text := '  !'
                        +  '    Clipboard  BitMap';
  end;
end;

// ========================================================================
//               END OF IMPLEMENTATION
// ========================================================================
end.
